using System;
using System.Text;
using System.Data;
using System.Collections;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// The class provides a functionality of serializing .NET collections and complex data structures
	/// into M-parseable format (concatenated strings, with a '^' used as a separator). 
	/// </summary>
	public class VistAMumpsFormatter
	{
		private const char ItemSeparatorChar = '^';
		private const char RowTerminator = '\n';
		private const string NullValueSubstitute = "";

		/// <summary>
		/// Exception used internally to notify caller up a call stack of disallowed character found
		/// in item to convert.
		/// </summary>
		private class DelimiterCharacterEncounteredException : VistALinkException
		{
			public DelimiterCharacterEncounteredException( string message )
				: base( message ) {}
		}

		/// <summary>
		/// The class should not be initialized.
		/// </summary>
		private VistAMumpsFormatter() {}

		/// <summary>
		/// Converts supplied object implementing <see cref="IEnumerable"/> interface into 
		/// M-parseable string, concatenating items together with '^' as a separator and 
		/// dumping results into supplied <see cref="StringBuilder"/>.
		/// Values contained in supplied collection cannot contain '^' and '\n' characters.
		/// </summary>
		/// <remarks>
		///		Important: this method does not check input parameters for performace sake.
		///		Caller has to make sure all parameters are correct.
		/// </remarks>
		/// <param name="sb"><see cref="StringBuilder"/> to write results to.</param>
		/// <param name="sourceItems">
		///		Collection of items implementing <see cref="IEnumerable"/> interface to be
		///		serialized to M-parseable format.
		/// </param>
		private static void DumpCollectionToMArrayViaStringBuilder( StringBuilder sb, IEnumerable sourceItems )
		{
			IEnumerator _ptr = sourceItems.GetEnumerator();

			if( !_ptr.MoveNext() )
			{
				sb.Append( NullValueSubstitute );
				return;
			}

			sb.Append( ConvertToAddToMParseableString( _ptr.Current ) );
	
			while( _ptr.MoveNext() )
			{
				sb.Append( ItemSeparatorChar );
				sb.Append( ConvertToAddToMParseableString( _ptr.Current ) );
			}
		}

		/// <summary>
		/// Converts supplied object to string ready to be put in M-parseable array.
		/// Checks that string does not contain delimiter chars '^' and '\n', if these
		/// characters are encountered - throws an exception. 
		/// </summary>
		/// <param name="objectToConvert">Object to convert.</param>
		/// <returns>String ready to be put into M array.</returns>
		private static string ConvertToAddToMParseableString( object objectToConvert )
		{
			if( objectToConvert == null )
				return NullValueSubstitute;

			string _result = objectToConvert.ToString();
			int _disallowedCharIndex = -1;

			if( ( _disallowedCharIndex = _result.IndexOfAny( new char[]{ ItemSeparatorChar, RowTerminator } ) ) > 0 )
				throw( new DelimiterCharacterEncounteredException( SR.Exceptions.VistAMFormatterDisallowedCharEncountered( _result, _result[_disallowedCharIndex] ) ) );

			return _result;
		}

		/// <summary>
		/// Converts supplied object implementing <see cref="IEnumerable"/> interface into M-parseable string,
		/// concatenating items together with '^' as a separator.
		/// Values contained in supplied collection cannot contain '^' and '\n' characters.
		/// </summary>
		/// <param name="sourceItems">
		///		Collection of items implementing <see cref="IEnumerable"/> interface to be
		///		serialized to M-parseable format.
		/// </param>
		/// <returns>
		///		String representing contents of supplied <paramref name="sourceItems" /> parameter
		///		converted to M-parseable string.
		///	</returns>
		public static string ToMArray( IEnumerable sourceItems )
		{
			if( sourceItems == null )
				throw( new ArgumentNullException( "sourceItems" ) );

			StringBuilder _sb = new StringBuilder();

			try
			{
				DumpCollectionToMArrayViaStringBuilder( _sb, sourceItems );
			}
			catch( DelimiterCharacterEncounteredException xcp )
			{
				throw( new ArgumentException( xcp.Message, "sourceItems" ) );
			}

			return _sb.ToString();
		}

		/// <summary>
		/// Converts supplied <paramref name="sourceDataTable" /> <see cref="DataTable"/> to M-parseable format.
		/// Values contained in <see cref="DataTable"/> cannot contain '^' and '\n' characters.
		/// </summary>
		/// <param name="sourceDataTable"><see cref="DataTable"/> object to convert.</param>
		/// <returns>
		///		String representing contents of supplied <paramref name="sourceDataTable" /> parameter
		///		converted to M-parseable string.
		///	</returns>
		public static string ToMArray( DataTable sourceDataTable )
		{
			if( sourceDataTable == null )
				throw( new ArgumentNullException( "sourceDataTable" ) );

			StringBuilder _sb = new StringBuilder();

			try
			{
				DumpCollectionToMArrayViaStringBuilder( _sb, sourceDataTable.Rows[0].ItemArray );

				for( int _rowIndex = 1; _rowIndex < sourceDataTable.Rows.Count; _rowIndex++ )
				{
					_sb.Append( RowTerminator );
					DumpCollectionToMArrayViaStringBuilder( _sb, sourceDataTable.Rows[_rowIndex].ItemArray );
				}
			}
			catch( DelimiterCharacterEncounteredException xcp )
			{
				throw( new ArgumentException( xcp.Message, "sourceDataTable" ) );
			}

			return _sb.ToString();
		}
	}
}
